// app.js

async function fetchJSON(url, options={}){
  const res = await fetch(url, options);
  if(!res.ok) throw new Error(await res.text());
  return res.json();
}

if (location.pathname === '/' || location.pathname.endsWith('/public/') || location.pathname.endsWith('/public/index.php')) {
  // Public booking
  const svcSel = document.getElementById('service');
  const dateInput = document.getElementById('date');
  const slotsDiv = document.getElementById('slots');
  const form = document.getElementById('bookForm');
  const msg = document.getElementById('msg');
  const cancelBtn = document.getElementById('cancelBtn');
  let selectedSlot = null;

  (async () => {
    const svcs = await fetchJSON('/api/services.php');
    svcSel.innerHTML = svcs.map(s=>`<option value="${s.id}" data-dur="${s.duration_minutes}">${s.name} (${s.duration_minutes} min)</option>`).join('');
    const today = new Date().toISOString().slice(0,10);
    dateInput.value = today;
    loadSlots();
  })();

  async function loadSlots(){
    slotsDiv.innerHTML = 'Beschikbaarheid laden...';
    selectedSlot = null; form.classList.add('hidden');
    const date = dateInput.value;
    const service_id = svcSel.value;
    if(!date || !service_id){ slotsDiv.innerHTML=''; return; }
    try {
      const data = await fetchJSON(`/api/availability.php?date=${date}&service_id=${service_id}`);
      if (!data.slots.length) { slotsDiv.innerHTML = '<em>Geen tijden beschikbaar</em>'; return; }
      slotsDiv.innerHTML = '';
      data.slots.forEach(s=>{
        const b = document.createElement('button');
        b.type='button'; b.className='slot'; b.textContent = `${s.start} – ${s.end}`;
        b.addEventListener('click', ()=>{
          document.querySelectorAll('.slot.selected').forEach(el=>el.classList.remove('selected'));
          b.classList.add('selected');
          selectedSlot = s;
          form.classList.remove('hidden');
        });
        slotsDiv.appendChild(b);
      });
    } catch(e){ slotsDiv.innerHTML = '<em>Fout bij laden</em>'; }
  }

  svcSel?.addEventListener('change', loadSlots);
  dateInput?.addEventListener('change', loadSlots);
  cancelBtn?.addEventListener('click', ()=>{ selectedSlot=null; form.classList.add('hidden'); document.querySelectorAll('.slot.selected').forEach(el=>el.classList.remove('selected')); });

  form?.addEventListener('submit', async (e)=>{
    e.preventDefault();
    if (!selectedSlot) return;
    const f = new FormData(form);
    const payload = {
      full_name: f.get('full_name'),
      email: f.get('email'),
      notes: f.get('notes'),
      service_id: parseInt(svcSel.value,10),
      date: document.getElementById('date').value,
      time: selectedSlot.start
    };
    msg.textContent = 'Bezig met boeken...';
    try{
      const res = await fetchJSON('/api/appointments.php', {
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body: JSON.stringify(payload)
      });
      msg.textContent = '✅ Afspraak bevestigd!';
    }catch(e){
      msg.textContent = '❌ Kon niet boeken: ' + e.message;
    }
  });
}

// Admin JS (loads when on /admin/ because HTML includes same app.js)
if (location.pathname.startsWith('/admin')) {
  const agendaDate = document.getElementById('agendaDate');
  const agendaList = document.getElementById('agendaList');
  const svcList = document.getElementById('svcList');
  const hoursList = document.getElementById('hoursList');
  const custList = document.getElementById('custList');

  // Agenda
  if (agendaDate) {
    agendaDate.value = new Date().toISOString().slice(0,10);
    document.getElementById('refreshAgenda').addEventListener('click', loadAgenda);
    loadAgenda();
  }

  async function loadAgenda(){
    agendaList.innerHTML = 'Laden...';
    const date = agendaDate.value;
    const items = await fetchJSON('/api/appointments.php?date='+date);
    agendaList.innerHTML = items.map(i=>`
      <div class="glass" style="margin:8px 0;padding:12px;">
        <b>${new Date(i.start_time).toLocaleString()}</b> – ${new Date(i.end_time).toLocaleTimeString([], {hour:'2-digit',minute:'2-digit'})}
        <div>${i.full_name} · ${i.service}</div>
        <div class="row">
          <button data-action="del" data-id="${i.id}" class="secondary">Verwijderen</button>
        </div>
      </div>`).join('');
    agendaList.querySelectorAll('button[data-action="del"]').forEach(btn=>{
      btn.addEventListener('click', async ()=>{
        if (!confirm('Verwijderen?')) return;
        await fetch('/api/appointments.php?id='+btn.dataset.id, {method:'DELETE'});
        loadAgenda();
      });
    });
  }

  // Services
  if (svcList) {
    document.getElementById('svcAdd').addEventListener('click', async ()=>{
      const name = document.getElementById('svcName').value.trim();
      const dur = parseInt(document.getElementById('svcDur').value,10);
      if(!name || !dur) return;
      await fetch('/api/services.php', {method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({name, duration_minutes:dur})});
      document.getElementById('svcName').value=''; document.getElementById('svcDur').value='';
      loadServices();
    });
    loadServices();
  }

  async function loadServices(){
    const svcs = await fetchJSON('/api/services.php');
    svcList.innerHTML = svcs.map(s=>`
      <div class="glass" style="margin:8px 0;padding:12px;">
        <b>${s.name}</b> • ${s.duration_minutes} min
        <div class="row">
          <button class="secondary" data-edit="${s.id}">Wijzig</button>
          <button data-del="${s.id}">Verwijder</button>
        </div>
      </div>`).join('');
    svcList.querySelectorAll('button[data-del]').forEach(b=>b.addEventListener('click', async ()=>{
      if(!confirm('Verwijderen?')) return;
      await fetch('/api/services.php?id='+b.dataset.del, {method:'DELETE'});
      loadServices();
    }));
    svcList.querySelectorAll('button[data-edit]').forEach(b=>b.addEventListener('click', async ()=>{
      const id = b.dataset.edit;
      const name = prompt('Nieuwe naam:');
      if (!name) return;
      const dur = parseInt(prompt('Nieuwe duur (minuten):')||'0',10);
      if (!dur) return;
      await fetch('/api/services.php?id='+id, {method:'PUT', headers:{'Content-Type':'application/json'}, body: JSON.stringify({name, duration_minutes:dur})});
      loadServices();
    }));
  }

  // Working hours
  if (hoursList) {
    loadHours();
    document.getElementById('hoursSave').addEventListener('click', saveHours);
  }
  async function loadHours(){
    const days = ['Zondag','Maandag','Dinsdag','Woensdag','Donderdag','Vrijdag','Zaterdag'];
    const rows = await fetchJSON('/api/working_hours.php');
    hoursList.innerHTML = rows.map(r=>`
      <div class="row" data-weekday="${r.weekday}">
        <b style="width:120px">${days[r.weekday]}</b>
        <label style="flex:1">Open <input type="time" value="${r.open_time}" class="open"></label>
        <label style="flex:1">Dicht <input type="time" value="${r.close_time}" class="close"></label>
        <label><input type="checkbox" class="closed" ${parseInt(r.is_closed)?'checked':''}> Gesloten</label>
      </div>`).join('');
  }
  async function saveHours(){
    const rows = Array.from(hoursList.querySelectorAll('.row')).map(r=>({ 
      weekday: parseInt(r.dataset.weekday,10),
      open_time: r.querySelector('.open').value,
      close_time: r.querySelector('.close').value,
      is_closed: r.querySelector('.closed').checked ? 1 : 0
    }));
    await fetch('/api/working_hours.php', {method:'PUT', headers:{'Content-Type':'application/json'}, body: JSON.stringify({rows})});
    alert('Opgeslagen');
  }

  // Customers & email summary
  const custSearch = document.getElementById('custSearch');
  if (custSearch && custList) {
    custSearch.addEventListener('input', loadCustomers);
    loadCustomers();
  }
  async function loadCustomers(){
    const q = custSearch.value.trim();
    const data = await fetchJSON('/api/customers.php'+(q?('?q='+encodeURIComponent(q)):'') );
    custList.innerHTML = data.map(c=>`<div class="glass" style="margin:6px 0;padding:10px"><b>${c.full_name}</b> · ${c.email}</div>`).join('');
  }
  const sendSummary = document.getElementById('sendSummary');
  if (sendSummary) {
    sendSummary.addEventListener('click', async ()=>{
      const email = document.getElementById('emailSummary').value.trim();
      if (!email) return;
      await fetch('/api/email_summary.php', {method:'POST', headers:{'Content-Type':'application/json'}, body: JSON.stringify({email})});
      alert('Verzonden (indien e-mail correct is ingesteld).');
    });
  }
}
